<?php
include 'connection.php';
include 'decryption_api_key.php';
require_once('/home/probash3/private/privet_key.php');
require_once('/home/probash3/private/encryption_helper.php');

header('Content-Type: application/json');
date_default_timezone_set('Asia/Dhaka');

$requestData = json_decode(file_get_contents("php://input"), true);
$apiKeyValidation = validateApiKey($requestData['api_key'] ?? null);

if (!$apiKeyValidation['status']) {
    echo json_encode(['status' => 'error', 'message' => $apiKeyValidation['message']]);
    $conn->close();
    exit();
}

// Encrypted inputs
$encrypted_user_id = $requestData['id'] ?? '';
$encrypted_amount  = $requestData['amount'] ?? '';
$encrypted_item_id = $requestData['item_ID'] ?? '';
$encrypted_trx_id  = $requestData['Trx_ID'] ?? '';
$encrypted_pin     = $requestData['PIN'] ?? '';

// Validate presence
if (empty($encrypted_user_id) || empty($encrypted_amount) || empty($encrypted_item_id) || empty($encrypted_pin)) {
    echo json_encode(['status' => 'error', 'message' => 'User ID, Amount, Item ID, and PIN are required']);
    $conn->close();
    exit();
}

// Decrypt data
$user_id = decryptData($encrypted_user_id);
$amount  = decryptData($encrypted_amount);
$item_id = decryptData($encrypted_item_id);
$tx_id   = decryptData($encrypted_trx_id);
$pin     = decryptData($encrypted_pin);

// Fallback if decryption fails
if (empty($tx_id)) {
    $tx_id = uniqid("TRX_", true);
}

// Sanity check
if (empty($user_id) || empty($amount) || empty($item_id) || empty($pin) || !is_numeric($amount) || $amount <= 0) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid or failed to decrypt required data']);
    $conn->close();
    exit();
}

// Verify user details
$userSql = "SELECT PIN, Balance, Veryfi_Status FROM USER_DATABSE_PIN WHERE id = ?";
$stmt = $conn->prepare($userSql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows == 0) {
    echo json_encode(['status' => 'error', 'message' => 'User not found']);
    $stmt->close();
    $conn->close();
    exit();
}

$row = $result->fetch_assoc();

// 🔑 Decrypt stored PIN and Balance
$stored_pin      = decryptSensitive($row['PIN']);
$current_balance = (float) decryptSensitive($row['Balance']);
$veryfi_status   = $row['Veryfi_Status'];

$stmt->close();

// Check verification and PIN match
if ($veryfi_status != 1) {
    echo json_encode(['status' => 'error', 'message' => 'Account is not verified']);
    $conn->close();
    exit();
}

if ($stored_pin !== $pin) {
    echo json_encode(['status' => 'error', 'message' => 'Incorrect PIN']);
    $conn->close();
    exit();
}

// Check and update balance
if ($current_balance < $amount) {
    echo json_encode(['status' => 'error', 'message' => 'Insufficient balance']);
    $conn->close();
    exit();
}

$updated_balance = $current_balance - $amount;

// 🔒 Encrypt new balance before saving
$encrypted_new_balance = encryptSensitive($updated_balance);

$updateSql = "UPDATE USER_DATABSE_PIN SET Balance = ? WHERE id = ?";
$stmtUpdate = $conn->prepare($updateSql);
$stmtUpdate->bind_param("si", $encrypted_new_balance, $user_id);

if (!$stmtUpdate->execute()) {
    error_log("Failed to update balance: " . $stmtUpdate->error);
    echo json_encode(['status' => 'error', 'message' => 'Balance deduction failed']);
    $stmtUpdate->close();
    $conn->close();
    exit();
}
$stmtUpdate->close();

// Insert purchase record
$purchaseSql = "INSERT INTO VPN_Card_Purchesh_History (User_ID, Item_ID, Item_Amount, Status, Date, Details_of_Item, Trx_ID)
                VALUES (?, ?, ?, ?, ?, ?, ?)";

$stmtPurchase = $conn->prepare($purchaseSql);
$status = "Pending";
$current_date = date("Y-m-d H:i:s");
$details_of_item = "NULL";

$stmtPurchase->bind_param("iisssss", $user_id, $item_id, $amount, $status, $current_date, $details_of_item, $tx_id);

if ($stmtPurchase->execute()) {
    echo json_encode(['status' => 'success', 'message' => 'Purchase successful']);
    
    // 🔔 Notify all admins via FCM
    require 'get-access-token.php';
    try {
        $adminTokenSql = "SELECT FCM_Token FROM Admin_Login WHERE LENGTH(FCM_Token) > 100";
        $adminTokenResult = $conn->query($adminTokenSql);

        if ($adminTokenResult->num_rows > 0) {
            $accessToken = getAccessToken('service-account-file.json');

            $title = "New VPN Card Purchase request";
            $body = "৳" . number_format($amount, 2) . " spent on VPN Card (Item ID: {$item_id}) — TRX ID: {$tx_id}";

            $headers = [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken
            ];

            while ($adminRow = $adminTokenResult->fetch_assoc()) {
                $fcm_token = $adminRow['FCM_Token'];

                $payload = json_encode([
                    'message' => [
                        'token' => $fcm_token,
                        'data' => [
                            'title' => $title,
                            'body'  => $body,
                            'type'  => 'vpn_card_buy'
                        ]
                    ]
                ]);

                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/pinshop-96581/messages:send");
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                $fcmResponse = curl_exec($ch);
                if ($fcmResponse === FALSE) {
                    error_log("FCM Send Error: " . curl_error($ch));
                } else {
                    error_log("FCM Response: " . $fcmResponse);
                }

                curl_close($ch);
            }
        }
    } catch (Exception $e) {
        error_log("FCM Notification Error: " . $e->getMessage());
    }
    
} else {
    error_log("Failed to insert purchase record: " . $stmtPurchase->error);
    echo json_encode(['status' => 'error', 'message' => 'Purchase processed, but history logging failed']);
}

$stmtPurchase->close();
$conn->close();
?>