<?php
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('Asia/Dhaka');

include 'connection.php';
include 'decryption_api_key.php';
require_once('/home/probash3/private/privet_key.php');
require_once('/home/probash3/private/encryption_helper.php');
require 'get-access-token.php';

$requestData = json_decode(file_get_contents("php://input"), true);

if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON format']);
    exit();
}

error_log("Raw JSON Data Received: " . json_encode($requestData));

// Step 1: Validate API key
$apiKeyValidation = validateApiKey($requestData['api_key'] ?? null);
if (!$apiKeyValidation['status']) {
    echo json_encode(['status' => 'error', 'message' => $apiKeyValidation['message']]);
    exit();
}

// Step 2: Decrypt User ID
$encrypted_user_id = $requestData['id'] ?? '';
if (empty($encrypted_user_id)) {
    echo json_encode(['status' => 'error', 'message' => 'User ID is required']);
    exit();
}

$user_id = decryptData($encrypted_user_id);
if (empty($user_id)) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to decrypt User ID']);
    exit();
}

// Step 3: Decrypt reward fields
$encrypted_point         = $requestData['Point'] ?? '';
$encrypted_phone_number  = $requestData['Phone_number'] ?? '';
$encrypted_amount        = $requestData['Amount'] ?? '';
$encrypted_trx_id        = $requestData['Trx_id'] ?? '';
$encrypted_item_id       = $requestData['Iteam_id'] ?? '';

if (empty($encrypted_point) || empty($encrypted_phone_number) || empty($encrypted_amount) || empty($encrypted_trx_id) || empty($encrypted_item_id)) {
    echo json_encode(['status' => 'error', 'message' => 'Required fields are missing']);
    exit();
}

$point        = decryptData($encrypted_point);
$phone_number = decryptData($encrypted_phone_number);
$amount       = decryptData($encrypted_amount);
$trx_id       = decryptData($encrypted_trx_id);
$item_id      = decryptData($encrypted_item_id);

if (empty($point) || empty($phone_number) || empty($amount) || empty($trx_id) || empty($item_id)) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to decrypt necessary data']);
    exit();
}

// Step 4: Get Reward and Veryfi_Status
$sql = "SELECT Reward, Veryfi_Status FROM USER_DATABSE_PIN WHERE id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$stmt->close();

if ($result->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'User not found']);
    exit();
}

$userData = $result->fetch_assoc();

// 🔑 Decrypt Reward before checking
$current_reward = (float) decryptSensitive($userData['Reward']);
$veryfi_status  = $userData['Veryfi_Status'];

if ($veryfi_status != 1) {
    echo json_encode(['status' => 'error', 'message' => 'Please verify your account before submitting recharge request']);
    exit();
}

if ($current_reward < $point) {
    echo json_encode(['status' => 'error', 'message' => 'Insufficient reward points. Earn more transactions daily to increase your balance.']);
    exit();
}

// Step 5: Update user reward balance
$new_reward_balance = $current_reward - $point;

// 🔒 Encrypt new reward balance before saving
$encrypted_new_reward = encryptSensitive($new_reward_balance);

$updateSql = "UPDATE USER_DATABSE_PIN SET Reward = ? WHERE id = ?";
$stmtUpdate = $conn->prepare($updateSql);
$stmtUpdate->bind_param("si", $encrypted_new_reward, $user_id);

if (!$stmtUpdate->execute()) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to update reward balance']);
    exit();
}
$stmtUpdate->close();

// Step 6: Insert request record
$request_date   = date("Y-m-d H:i:s");
$request_status = "Pending";

$insertSql = "INSERT INTO Reward_Recharge_Request (User_ID, Request_number, Total_Point, Amount_Recharge, Status, Date, Trx_ID) 
              VALUES (?, ?, ?, ?, ?, ?, ?)";
$stmtInsert = $conn->prepare($insertSql);
$stmtInsert->bind_param("issssss", $user_id, $phone_number, $point, $amount, $request_status, $request_date, $trx_id);

if ($stmtInsert->execute()) {
    echo json_encode(['status' => 'success', 'message' => 'Reward recharge request submitted successfully']);

    // 🔔 Send FCM notification to all admins
    try {
        $adminTokenSql = "SELECT FCM_Token FROM Admin_Login WHERE LENGTH(FCM_Token) > 100";
        $adminTokenResult = $conn->query($adminTokenSql);

        if ($adminTokenResult->num_rows > 0) {
            $accessToken = getAccessToken('service-account-file.json');

            $title = "New Reward Recharge Request";
            $body  = "User requested ৳" . number_format($amount, 2) .
                     " recharge using " . number_format($point, 0) . " points (Trx ID: $trx_id).";

            $headers = [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken
            ];

            while ($adminRow = $adminTokenResult->fetch_assoc()) {
                $fcm_token = $adminRow['FCM_Token'];

                $payload = json_encode([
                    'message' => [
                        'token' => $fcm_token,
                        'notification' => [
                            'title' => $title,
                            'body'  => $body
                        ]
                    ]
                ]);

                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/pinshop-96581/messages:send");
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                $fcmResponse = curl_exec($ch);
                if ($fcmResponse === FALSE) {
                    error_log("FCM Send Error: " . curl_error($ch));
                } else {
                    error_log("FCM Response: " . $fcmResponse);
                }

                curl_close($ch);
            }
        }
    } catch (Exception $e) {
        error_log("FCM Notification Error: " . $e->getMessage());
    }

} else {
    echo json_encode(['status' => 'error', 'message' => 'Failed to submit reward recharge request']);
}

$stmtInsert->close();
$conn->close();
?>