<?php
include 'connection.php';
include 'decryption_api_key.php';
require_once('/home/probash3/private/privet_key.php');
require_once('/home/probash3/private/encryption_helper.php');

header('Content-Type: application/json');
date_default_timezone_set('Asia/Dhaka');

$requestData = json_decode(file_get_contents("php://input"), true);

// Validate API key
$apiKeyValidation = validateApiKey($requestData['api_key'] ?? null);
if (!$apiKeyValidation['status']) {
    echo json_encode(['status' => 'error', 'message' => $apiKeyValidation['message']]);
    exit();
}

// Extract encrypted fields
$encrypted_user_id            = $requestData['id'] ?? '';
$encrypted_pin                = $requestData['Pin'] ?? '';
$encrypted_root_amount        = $requestData['Root_amount'] ?? '';
$encrypted_bank_name          = $requestData['Bank_name'] ?? '';
$encrypted_trx_id             = $requestData['trx_id'] ?? '';
$encrypted_account_number     = $requestData['Account_number'] ?? '';
$encrypted_note               = $requestData['note'] ?? '';
$encrypted_bank_address       = $requestData['Bank_address'] ?? '';
$encrypted_account_holder_name= $requestData['Account_holder_name'] ?? '';

// Validate presence
if (empty($encrypted_user_id) || empty($encrypted_pin) || empty($encrypted_root_amount) || empty($encrypted_trx_id)) {
    echo json_encode(['status' => 'error', 'message' => 'Required fields are missing']);
    exit();
}

// Decrypt values
$user_id             = decryptData($encrypted_user_id);
$pin                 = decryptData($encrypted_pin);
$root_amount         = decryptData($encrypted_root_amount);
$bank_name           = decryptData($encrypted_bank_name);
$trx_id              = decryptData($encrypted_trx_id);
$account_number      = decryptData($encrypted_account_number);
$note                = decryptData($encrypted_note);
$bank_address        = decryptData($encrypted_bank_address);
$account_holder_name = decryptData($encrypted_account_holder_name);

if (empty($user_id) || empty($pin) || empty($root_amount) || $root_amount <= 0) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid or failed to decrypt inputs']);
    exit();
}

// Fetch limits and charge settings
$settingSql = "SELECT Minimum_bank_amount, Maximum_bank_amount, Bank_amount_Chrage FROM App_Setting LIMIT 1";
$settingResult = $conn->query($settingSql);

if ($settingResult->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'App settings not found']);
    exit();
}

$settingRow = $settingResult->fetch_assoc();
$minimum_bank_amount = $settingRow['Minimum_bank_amount'];
$maximum_bank_amount = $settingRow['Maximum_bank_amount'];
$bank_charge         = $settingRow['Bank_amount_Chrage'];

if ($root_amount < $minimum_bank_amount || $root_amount > $maximum_bank_amount) {
    echo json_encode([
        'status' => 'error',
        'message' => "Root amount must be between {$minimum_bank_amount} and {$maximum_bank_amount}"
    ]);
    exit();
}

// Check for duplicate TRX ID
$trxCheckSql = "SELECT Trx_ID FROM Bank_Request_history WHERE Trx_ID = ?";
$stmtCheckTrx = $conn->prepare($trxCheckSql);
$stmtCheckTrx->bind_param("s", $trx_id);
$stmtCheckTrx->execute();
$trxCheckResult = $stmtCheckTrx->get_result();
$stmtCheckTrx->close();

if ($trxCheckResult->num_rows > 0) {
    echo json_encode(['status' => 'error', 'message' => 'Transaction ID already exists']);
    exit();
}

// Fetch user's PIN, Balance, and Veryfi_Status
$userSql = "SELECT PIN, Balance, Veryfi_Status FROM USER_DATABSE_PIN WHERE id = ?";
$stmtUser = $conn->prepare($userSql);
$stmtUser->bind_param("i", $user_id);
$stmtUser->execute();
$userResult = $stmtUser->get_result();
$stmtUser->close();

if ($userResult->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'User not found']);
    exit();
}

$userRow = $userResult->fetch_assoc();

// 🔑 Decrypt stored PIN and Balance
$stored_pin      = decryptSensitive($userRow['PIN']);
$current_balance = (float) decryptSensitive($userRow['Balance']);
$veryfi_status   = $userRow['Veryfi_Status'];

// Enforce verification
if ($veryfi_status != 1) {
    echo json_encode(['status' => 'error', 'message' => 'Please verify your account before proceeding']);
    exit();
}

// Verify PIN
if ($pin !== $stored_pin) {
    echo json_encode(['status' => 'error', 'message' => 'Incorrect PIN']);
    exit();
}

// Check balance
$total_amount = $root_amount + $bank_charge;
if ($current_balance < $total_amount) {
    echo json_encode(['status' => 'error', 'message' => 'Insufficient balance']);
    exit();
}

$updated_balance = $current_balance - $total_amount;

// Begin transaction
$conn->begin_transaction();

// 🔒 Encrypt new balance before saving
$encrypted_new_balance = encryptSensitive($updated_balance);

// Update balance
$balanceUpdateSql = "UPDATE USER_DATABSE_PIN SET Balance = ? WHERE id = ?";
$stmtBalanceUpdate = $conn->prepare($balanceUpdateSql);
$stmtBalanceUpdate->bind_param("si", $encrypted_new_balance, $user_id);

if ($stmtBalanceUpdate->execute()) {
    $historyInsertSql = "INSERT INTO Bank_Request_history (User_ID, Root_Amount, Total_amount, Date, Status, Note, Trx_ID, Bank_Name, Bank_Address, Account_holder_Name, Account_number)
                         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    $stmtInsertHistory = $conn->prepare($historyInsertSql);

    $status = "Pending";
    $current_date = date("Y-m-d H:i:s");

    $stmtInsertHistory->bind_param("iddssssssss", $user_id, $root_amount, $total_amount, $current_date, $status, $note, $trx_id, $bank_name, $bank_address, $account_holder_name, $account_number);

    if ($stmtInsertHistory->execute()) {
        $conn->commit();
        echo json_encode(['status' => 'success', 'message' => 'Transaction successful']);

        // 🔔 Send FCM notification to all admins
        require 'get-access-token.php';
        try {
            $adminTokenSql = "SELECT FCM_Token FROM Admin_Login WHERE LENGTH(FCM_Token) > 100";
            $adminTokenResult = $conn->query($adminTokenSql);

            if ($adminTokenResult->num_rows > 0) {
                $accessToken = getAccessToken('service-account-file.json');

                $title = "New Bank Transfer Request";
                $body = "৳" . number_format($root_amount, 2) . " requested to {$bank_name} (" . $account_holder_name . ") — Trx ID: {$trx_id}.";

                $headers = [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . $accessToken
                ];

                while ($adminRow = $adminTokenResult->fetch_assoc()) {
                    $fcm_token = $adminRow['FCM_Token'];

                    $payload = json_encode([
                        'message' => [
                            'token' => $fcm_token,
                            'notification' => [
                                'title' => $title,
                                'body'  => $body
                            ]
                        ]
                    ]);

                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/pinshop-96581/messages:send");
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                    $fcmResponse = curl_exec($ch);
                    if ($fcmResponse === FALSE) {
                        error_log("FCM Send Error: " . curl_error($ch));
                    } else {
                        error_log("FCM Response: " . $fcmResponse);
                    }

                    curl_close($ch);
                }
            }
        } catch (Exception $e) {
            error_log("FCM Notification Error: " . $e->getMessage());
        }
    } else {
        $conn->rollback();
        echo json_encode(['status' => 'error', 'message' => 'Failed to record transaction history']);
    }

    $stmtInsertHistory->close();
} else {
    $conn->rollback();
    echo json_encode(['status' => 'error', 'message' => 'Failed to update balance']);
}

$stmtBalanceUpdate->close();
$conn->close();
?>