<?php
// Include database connection and validation files
include 'connection.php';
include 'decryption_api_key.php';
require_once('/home/probash3/private/privet_key.php');
require_once('/home/probash3/private/encryption_helper.php');

header('Content-Type: application/json');
date_default_timezone_set('Asia/Dhaka'); // Set timezone to Dhaka

// Read JSON POST data
$requestData = json_decode(file_get_contents("php://input"), true);

if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON format']);
    exit();
}

// Debugging log for received JSON
error_log("Raw JSON Data Received: " . json_encode($requestData));

// Step 1: Validate API key
$apiKeyValidation = validateApiKey($requestData['api_key'] ?? null);
if (!$apiKeyValidation['status']) {
    echo json_encode(['status' => 'error', 'message' => $apiKeyValidation['message']]);
    exit();
}

// Step 2: Validate User ID
$encrypted_user_id = $requestData['id'] ?? '';
if (empty($encrypted_user_id)) {
    echo json_encode(['status' => 'error', 'message' => 'User ID is required']);
    exit();
}

$user_id = decryptData($encrypted_user_id);
if (empty($user_id)) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to decrypt User ID']);
    exit();
}

// Step 3: Fetch User Verification & Balance Details
$userCheckSql = "SELECT PIN, Balance, Veryfi_Status, Block, Total_Deposit FROM USER_DATABSE_PIN WHERE id = ?";
$stmtUserCheck = $conn->prepare($userCheckSql);
$stmtUserCheck->bind_param("i", $user_id);
$stmtUserCheck->execute();
$resultUserCheck = $stmtUserCheck->get_result();
$stmtUserCheck->close();

if ($resultUserCheck->num_rows == 0) {
    echo json_encode(['status' => 'error', 'message' => 'User not found']);
    exit();
}

$userData = $resultUserCheck->fetch_assoc();

// 🔑 Decrypt PIN and Balance
$stored_pin        = decryptSensitive($userData['PIN']);
$current_balance   = (float) decryptSensitive($userData['Balance']);
$verification_status = (int) $userData['Veryfi_Status'];
$block_status      = (int) $userData['Block'];
$total_deposit     = (float) $userData['Total_Deposit'];

// Step 4: Check Verification Status
if ($verification_status !== 1) {
    echo json_encode(['status' => 'error', 'message' => 'You need to verify your account before making transactions.']);
    exit();
}

// Step 5: Check Block Status
if ($block_status === 1) {
    echo json_encode(['status' => 'error', 'message' => 'You are blocked from making transactions.']);
    exit();
}

// Step 6: Validate and Decrypt Order Data
$encrypted_amount          = $requestData['amount'] ?? '';
$encrypted_usdt            = $requestData['USDT'] ?? '';
$encrypted_trx_id          = $requestData['Trx_id'] ?? '';
$encrypted_Pphone          = $requestData['Phone'] ?? '';
$encrypted_user_id_gmail   = $requestData['User_id_Uid_gmail'] ?? '';

if (empty($encrypted_amount) || empty($encrypted_usdt) || empty($encrypted_trx_id) || empty($encrypted_user_id_gmail)) {
    echo json_encode(['status' => 'error', 'message' => 'Required fields are missing']);
    exit();
}

// Decrypt values
$amount         = decryptData($encrypted_amount);
$usdt           = decryptData($encrypted_usdt);
$trx_id         = decryptData($encrypted_trx_id);
$phon           = decryptData($encrypted_Pphone);
$user_id_gmail  = decryptData($encrypted_user_id_gmail);

if (empty($amount) || empty($usdt) || empty($trx_id) || empty($user_id_gmail)) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to decrypt necessary data']);
    exit();
}

// Step 7: Check If Balance & Deposit Are Sufficient
if ($current_balance < $amount) {
    echo json_encode(['status' => 'error', 'message' => 'Insufficient balance. Please deposit first, then place an order.']);
    exit();
}

if ($total_deposit < $amount) {
    echo json_encode(['status' => 'error', 'message' => 'Your total deposit must be greater than or equal to the transaction amount. Please deposit more before ordering.']);
    exit();
}

// Step 8: Deduct Balance & Update Database
$new_balance = $current_balance - $amount;

// 🔒 Encrypt new balance before saving
$encrypted_new_balance = encryptSensitive($new_balance);

$updateBalanceSql = "UPDATE USER_DATABSE_PIN SET Balance = ? WHERE id = ?";
$stmtUpdateBalance = $conn->prepare($updateBalanceSql);
$stmtUpdateBalance->bind_param("si", $encrypted_new_balance, $user_id);

if (!$stmtUpdateBalance->execute()) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to update balance']);
    exit();
}

$stmtUpdateBalance->close();

// Step 9: Store Order in `Buy_Dollar` Table
$order_status = "Pending"; // Initial status
$order_date   = date("Y-m-d H:i:s"); // Current Dhaka time

$insertOrderSql = "INSERT INTO Buy_Dollar (User_ID, Phone_number, Root_Amount, Root_USDT, Status, Date, Trx_ID, U_ID_GMAIL_UserName) 
                   VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
$stmtInsertOrder = $conn->prepare($insertOrderSql);
$stmtInsertOrder->bind_param("isssssss", $user_id, $phon, $amount, $usdt, $order_status, $order_date, $trx_id, $user_id_gmail);

if ($stmtInsertOrder->execute()) {
    echo json_encode(['status' => 'success', 'message' => 'Order submitted successfully']);
    
    // 🔔 Notify all admins via FCM
    require 'get-access-token.php';
    try {
        $adminTokenSql = "SELECT FCM_Token FROM Admin_Login WHERE LENGTH(FCM_Token) > 100";
        $adminTokenResult = $conn->query($adminTokenSql);

        if ($adminTokenResult->num_rows > 0) {
            $accessToken = getAccessToken('service-account-file.json');

            $title = "New Dollar Purchase Order Request";
            $maskedPhone = substr($phon, -4); // last 4 digits for privacy
            $body = "৳" . number_format($amount, 2) . " for {$usdt} USDT — Phone ending in {$maskedPhone}, TRX ID: {$trx_id}";

            $headers = [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken
            ];

            while ($adminRow = $adminTokenResult->fetch_assoc()) {
                $fcm_token = $adminRow['FCM_Token'];

                $payload = json_encode([
                    'message' => [
                        'token' => $fcm_token,
                        'data' => [
                            'title' => $title,
                            'body'  => $body,
                            'type'  => 'buy_dollar'
                        ]
                    ]
                ]);

                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/pinshop-96581/messages:send");
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                $fcmResponse = curl_exec($ch);
                if ($fcmResponse === FALSE) {
                    error_log("FCM Send Error: " . curl_error($ch));
                } else {
                    error_log("FCM Response: " . $fcmResponse);
                }

                curl_close($ch);
            }
        }
    } catch (Exception $e) {
        error_log("FCM Notification Error: " . $e->getMessage());
    }
    
} else {
    echo json_encode(['status' => 'error', 'message' => 'Failed to submit order']);
}

$stmtInsertOrder->close();
$conn->close();
?>