<?php
include 'connection.php';
include 'decryption_api_key.php';
require_once('/home/probash3/private/privet_key.php');
require_once('/home/probash3/private/encryption_helper.php');

header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('Asia/Dhaka');

error_log("Raw POST Request: " . file_get_contents("php://input"));
$requestData = json_decode(file_get_contents("php://input"), true);

// Validate API key
$apiKeyValidation = validateApiKey($requestData['api_key'] ?? null);
if (!$apiKeyValidation['status']) {
    echo json_encode(['status' => 'error', 'message' => $apiKeyValidation['message']]);
    exit();
}

// Encrypted inputs
$encrypted_user_id       = $requestData['id'] ?? '';
$encrypted_pin           = $requestData['Pin'] ?? '';
$encrypted_amount        = $requestData['Total_amount'] ?? '';
$encrypted_trx_id        = $requestData['TRX_id'] ?? '';
$encrypted_phone         = $requestData['phone_number'] ?? '';
$encrypted_oparator      = $requestData['Oparator'] ?? '';
$encrypted_oparator_type = $requestData['Oparator_type'] ?? '';

// Basic validation
if (empty($encrypted_user_id) || empty($encrypted_pin) || empty($encrypted_amount) || empty($encrypted_trx_id)) {
    echo json_encode(['status' => 'error', 'message' => 'User ID, PIN, Amount, and TRX ID are required']);
    exit();
}

// Decrypt values from POST
$user_id        = decryptData($encrypted_user_id);
$pin            = decryptData($encrypted_pin);
$amount         = decryptData($encrypted_amount);
$trx_id         = decryptData($encrypted_trx_id);
$phone_number   = decryptData($encrypted_phone);
$oparator       = decryptData($encrypted_oparator);
$oparator_type  = decryptData($encrypted_oparator_type);

if (empty($user_id) || empty($pin) || empty($amount) || empty($trx_id) || !is_numeric($amount) || $amount <= 0) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid or failed to decrypt User ID, PIN, Amount, or TRX ID']);
    exit();
}

// Check app settings
$settingSql = "SELECT Recharge_ON_OFF, Recharge_Minimum_Amount, Recharge_Max_Amount FROM App_Setting LIMIT 1";
$settingResult = $conn->query($settingSql);

if ($settingResult->num_rows == 0) {
    echo json_encode(['status' => 'error', 'message' => 'App settings not found']);
    exit();
}

$appSettings = $settingResult->fetch_assoc();

if ($appSettings['Recharge_ON_OFF'] == 1) {
    echo json_encode(['status' => 'error', 'message' => 'Mobile recharge is currently disabled']);
    exit();
}

$min_amount = $appSettings['Recharge_Minimum_Amount'];
$max_amount = $appSettings['Recharge_Max_Amount'];

if ($amount < $min_amount || $amount > $max_amount) {
    echo json_encode([
        'status' => 'error',
        'message' => "Total amount must be between {$min_amount} and {$max_amount}"
    ]);
    exit();
}

// Check duplicate TRX ID
$sqlCheckTrx = "SELECT TRX_Id FROM Mobile_Recharge_History WHERE TRX_Id = ?";
$stmtCheckTrx = $conn->prepare($sqlCheckTrx);
$stmtCheckTrx->bind_param("s", $trx_id);
$stmtCheckTrx->execute();
$resultCheckTrx = $stmtCheckTrx->get_result();
$stmtCheckTrx->close();

if ($resultCheckTrx->num_rows > 0) {
    echo json_encode(['status' => 'error', 'message' => 'Transaction ID already exists']);
    exit();
}

// Fetch user PIN, balance, and Veryfi_Status (stored encrypted)
$sql = "SELECT PIN, Balance, Veryfi_Status FROM USER_DATABSE_PIN WHERE id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$stmt->close();

if ($result->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'User not found']);
    exit();
}

$row = $result->fetch_assoc();

// Decrypt stored PIN and Balance
$stored_pin      = decryptSensitive($row['PIN']);
$current_balance = (float) decryptSensitive($row['Balance']);
$veryfi_status   = $row['Veryfi_Status'];

// Enforce verification
if ($veryfi_status != 1) {
    echo json_encode(['status' => 'error', 'message' => 'Please verify your account before proceeding']);
    exit();
}

if ($pin !== $stored_pin) {
    echo json_encode(['status' => 'error', 'message' => 'Incorrect PIN']);
    exit();
}

if ($current_balance < $amount) {
    echo json_encode(['status' => 'error', 'message' => 'Insufficient balance']);
    exit();
}

$updated_balance = $current_balance - $amount;

// Begin transaction
$conn->begin_transaction();

// Update balance (re‑encrypt before saving)
$encrypted_new_balance = encryptSensitive($updated_balance);

$updateSql = "UPDATE USER_DATABSE_PIN SET Balance = ? WHERE id = ?";
$stmtUpdate = $conn->prepare($updateSql);
$stmtUpdate->bind_param("si", $encrypted_new_balance, $user_id);

if ($stmtUpdate->execute()) {
    $insertSql = "INSERT INTO Mobile_Recharge_History (User_ID, Phone_number, Total_amount, Oparator, Oparator_type, Date, TRX_Id, Status)
                  VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
    $stmtInsert = $conn->prepare($insertSql);

    $status = "Pending";
    $current_date = date("Y-m-d H:i:s");

    $stmtInsert->bind_param("isssssss", $user_id, $phone_number, $amount, $oparator, $oparator_type, $current_date, $trx_id, $status);

    if ($stmtInsert->execute()) {
        $conn->commit();
        echo json_encode(['status' => 'success', 'message' => 'Transaction successful']);
        
        // Notify admins via FCM (unchanged)
        require 'get-access-token.php';
        try {
            $adminTokenSql = "SELECT FCM_Token FROM Admin_Login WHERE LENGTH(FCM_Token) > 100";
            $adminTokenResult = $conn->query($adminTokenSql);

            if ($adminTokenResult->num_rows > 0) {
                $accessToken = getAccessToken('service-account-file.json');

                $title = "New Mobile Recharge Request";
                $maskedPhone = substr($phone_number, -4);
                $body = number_format($amount, 2) . " recharge to {$oparator} ({$oparator_type}) — Phone ending in {$maskedPhone}, TRX ID: {$trx_id}";

                $headers = [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . $accessToken
                ];

                while ($adminRow = $adminTokenResult->fetch_assoc()) {
                    $fcm_token = $adminRow['FCM_Token'];

                    $payload = json_encode([
                        'message' => [
                            'token' => $fcm_token,
                            'notification' => [
                                'title' => $title,
                                'body' => $body
                            ]
                        ]
                    ]);

                    $ch = curl_init();
                    curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/pinshop-96581/messages:send");
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
                    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                    $fcmResponse = curl_exec($ch);
                    if ($fcmResponse === FALSE) {
                        error_log("FCM Send Error: " . curl_error($ch));
                    } else {
                        error_log("FCM Response: " . $fcmResponse);
                    }

                    curl_close($ch);
                }
            }
        } catch (Exception $e) {
            error_log("FCM Notification Error: " . $e->getMessage());
        }
    } else {
        $conn->rollback();
        echo json_encode(['status' => 'error', 'message' => 'Transaction succeeded, but history logging failed']);
    }

    $stmtInsert->close();
} else {
    $conn->rollback();
    echo json_encode(['status' => 'error', 'message' => 'Failed to update balance']);
}

$stmtUpdate->close();
$conn->close();
?>