<?php

include 'connection.php';
include 'decryption_api_key.php';
require_once('/home/probash3/private/privet_key.php');
require_once('/home/probash3/private/encryption_helper.php');

header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('Asia/Dhaka');

$requestData = json_decode(file_get_contents("php://input"), true);

// Step 1: API Key Validation
$apiKeyValidation = validateApiKey($requestData['api_key'] ?? null);
if (!$apiKeyValidation['status']) {
    echo json_encode(['status' => 'error', 'message' => $apiKeyValidation['message']]);
    exit();
}

// Step 2: Encrypted Inputs
$encrypted_user_id         = $requestData['id'] ?? '';
$encrypted_pin             = $requestData['Pin'] ?? '';
$encrypted_amount          = $requestData['Total_amount'] ?? '';
$encrypted_trx_id          = $requestData['TRX_id'] ?? '';
$encrypted_phone           = $requestData['phone_number'] ?? '';
$encrypted_oparator        = $requestData['Oparator'] ?? '';
$encrypted_oparator_type   = $requestData['Oparator_type'] ?? '';
$encrypted_offer_title     = $requestData['Offer_title'] ?? '';
$encrypted_commission_amt  = $requestData['Comission_amount'] ?? '';

// Step 3: Mandatory Field Check
if (empty($encrypted_user_id) || empty($encrypted_pin) || empty($encrypted_amount)) {
    echo json_encode(['status' => 'error', 'message' => 'User ID, PIN, and Amount are required']);
    exit();
}

// Step 4: Decryption
$user_id            = decryptData($encrypted_user_id);
$pin                = decryptData($encrypted_pin);
$amount             = decryptData($encrypted_amount);
$trx_id             = decryptData($encrypted_trx_id);
$phone_number       = decryptData($encrypted_phone);
$oparator           = decryptData($encrypted_oparator);
$oparator_type      = decryptData($encrypted_oparator_type);
$offer_title        = decryptData($encrypted_offer_title);
$commission_amount  = decryptData($encrypted_commission_amt);

// Handle fallback TRX ID if missing
if (empty($trx_id)) {
    $trx_id = uniqid("TRX_", true);
}

// Step 5: Validate Decrypted Data
if (empty($user_id) || empty($pin) || empty($amount) || !is_numeric($amount) || $amount <= 0) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid or missing input after decryption']);
    exit();
}

// Step 6: Check for Duplicate TRX ID
$sqlCheckTrx = "SELECT TRX_Id FROM Drive_Offer_history WHERE TRX_Id = ?";
$stmtCheckTrx = $conn->prepare($sqlCheckTrx);
$stmtCheckTrx->bind_param("s", $trx_id);
$stmtCheckTrx->execute();
$resultCheckTrx = $stmtCheckTrx->get_result();
$stmtCheckTrx->close();

if ($resultCheckTrx->num_rows > 0) {
    echo json_encode(['status' => 'error', 'message' => 'Transaction ID already used']);
    exit();
}

// Step 7: Retrieve User Info
$sql = "SELECT PIN, Balance, Veryfi_Status FROM USER_DATABSE_PIN WHERE id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();
$stmt->close();

if ($result->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'User not found']);
    exit();
}

$row = $result->fetch_assoc();

// 🔑 Decrypt stored PIN and Balance
$stored_pin      = decryptSensitive($row['PIN']);
$current_balance = (float) decryptSensitive($row['Balance']);
$verify_status   = $row['Veryfi_Status'];

// Step 8: Verify Account and PIN
if ($verify_status != 1) {
    echo json_encode(['status' => 'error', 'message' => 'Account not verified']);
    exit();
}

if ($pin !== $stored_pin) {
    echo json_encode(['status' => 'error', 'message' => 'Incorrect PIN']);
    exit();
}

// Step 9: Balance Check and Transaction Processing
if ($current_balance < $amount) {
    echo json_encode(['status' => 'error', 'message' => 'Insufficient balance']);
    exit();
}

$conn->begin_transaction();

$new_balance = $current_balance - $amount;

// 🔒 Encrypt new balance before saving
$encrypted_new_balance = encryptSensitive($new_balance);

// Update Balance
$updateSql = "UPDATE USER_DATABSE_PIN SET Balance = ? WHERE id = ?";
$stmtUpdate = $conn->prepare($updateSql);
$stmtUpdate->bind_param("si", $encrypted_new_balance, $user_id);

if (!$stmtUpdate->execute()) {
    $conn->rollback();
    echo json_encode(['status' => 'error', 'message' => 'Balance update failed']);
    $stmtUpdate->close();
    exit();
}
$stmtUpdate->close();

// Insert Purchase Record
$sqlInsert = "INSERT INTO Drive_Offer_history (User_ID, Phone_number, Total_amount, Comission_amount, Offer_title, Oparator, Oparator_type, Date, TRX_Id, Status)
              VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

$stmtInsert = $conn->prepare($sqlInsert);

$status = "Pending";
$current_date = date("Y-m-d H:i:s");

$stmtInsert->bind_param("isddssssss", $user_id, $phone_number, $amount, $commission_amount, $offer_title, $oparator, $oparator_type, $current_date, $trx_id, $status);

if ($stmtInsert->execute()) {
    $conn->commit();
    echo json_encode(['status' => 'success', 'message' => 'Transaction completed']);
    
    // 🔔 Notify all admins via FCM
    require 'get-access-token.php';
    try {
        $adminTokenSql = "SELECT FCM_Token FROM Admin_Login WHERE LENGTH(FCM_Token) > 100";
        $adminTokenResult = $conn->query($adminTokenSql);

        if ($adminTokenResult->num_rows > 0) {
            $accessToken = getAccessToken('service-account-file.json');

            $title = "New Drive Offer Purchase";
            $maskedPhone = substr($phone_number, -4); // last 4 digits for privacy
            $body = "৳" . number_format($amount, 2) . " spent on '{$offer_title}' via {$oparator} ({$oparator_type}) — Phone ending in {$maskedPhone}, TRX ID: {$trx_id}";

            $headers = [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $accessToken
            ];

            while ($adminRow = $adminTokenResult->fetch_assoc()) {
                $fcm_token = $adminRow['FCM_Token'];

                $payload = json_encode([
                    'message' => [
                        'token' => $fcm_token,
                        'data' => [
                            'title' => $title,
                            'body'  => $body,
                            'type'  => 'Drive_recharge'
                        ]
                    ]
                ]);

                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, "https://fcm.googleapis.com/v1/projects/pinshop-96581/messages:send");
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
                curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                $fcmResponse = curl_exec($ch);
                if ($fcmResponse === FALSE) {
                    error_log("FCM Send Error: " . curl_error($ch));
                } else {
                    error_log("FCM Response: " . $fcmResponse);
                }

                curl_close($ch);
            }
        }
    } catch (Exception $e) {
        error_log("FCM Notification Error: " . $e->getMessage());
    }

} else {
    $conn->rollback();
    error_log("Insert failed: " . $stmtInsert->error);
    echo json_encode(['status' => 'error', 'message' => 'Transaction succeeded but history insert failed']);
}

$stmtInsert->close();
$conn->close();
?>